﻿using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using System.IO;
using System.Net;
using Inet.Viewer;
using Inet.Viewer.WinForms;
using Inet.Viewer.test.WinForms;
using System.Windows.Forms;


namespace Inet.Viewer.Data
{
    /// <summary>
    /// 
    /// </summary>
    [TestFixture]
    public class TestURLRenderData
    {
        const string validUrl = "http://dell28:9000/?report=repo:/test/DotNetViewer4Pages.rpt";
        private const string validServerUrlWithParameters = "http://dell28:9000/?report=repo:%2Fcodereview_overview_old2.rpt&hasgrouptree=false&promptonrefresh=true";
        const string validUrl2 =                     "http://dell28:9000/?report=repo:%2Fstatistik%2Fmilestonetickets.rpt";
        const string validUrlIP4 =                   "http://210.1.164.110:9000/?report=repo:%2Fcodereview_overview_old2.rpt";
        const string inValidUrl =                    "http://";
        const string inValidReport =                 "http://dell28:9000/?report=repo:%2FsVeryRandomReportThatCannotEcist.rpt";
        const string validSubreportOnDemand =       "http://dell28:9000/?init=html&report=repo%3a%2ftest%2ffeature%2freports%2fsubreports%2fsubreport3.rpt&locale=de&subreport_ondemand=type%3d0%26subreport%3d0";

        /// <summary>
        /// 
        /// </summary>
        [Test]
        public void TestInvalidUrl()
        {
            string url;
            IRenderData data;
            try
            {
                url = inValidUrl;
                data = new URLRenderData(url);
                Assert.Fail();
            }
            catch (Exception)
            {
                /*if (e.InnerException != null && e.InnerException.GetType() == typeof(UriFormatException))
                {
                    
                }*/
                //Assert.Fail("Another Exception was thrown"  e);
                Assert.Pass("An Exception should be thrown");
            }
            
            url = inValidReport;
            data = new URLRenderData(url);
        }

        /// <summary>
        /// 
        /// </summary>
        [Test]
        public void TestReportSetParameters()
        {
            // with Url in Constructor
            IRenderData data = new URLRenderData(validUrl);            
            data["KeyOne"] = "ValueOne";
            data["KeyTwo"] = "ValueTwo";

            Assert.AreEqual("ValueOne", data["KeyOne"], "");
            Assert.AreEqual("ValueTwo", data["KeyTwo"],  "");
            
            // With Url set via ReportLocation
            data = new URLRenderData();
            data.ReportLocation = validUrl;
            data["KeyOne"] = "ValueOne";
            data["KeyTwo"] = "ValueTwo";
            
            // with double adding                        
            data["KeyOne"] = "ValueDoubleOne";

            Assert.AreEqual("ValueDoubleOne", data["KeyOne"], "Double adding");
            Assert.AreEqual("ValueTwo", data["KeyTwo"], " Not changed");
        
        }     

        /// <summary>
        /// 
        /// </summary>
        [Test]
        public void TestCloning()
        {
            IRenderData dataOriginal = new URLRenderData(validUrl);
            IRenderData dataClone = dataOriginal.Clone() as IRenderData;

            // Without params
            Assert.AreEqual(dataOriginal.ReportLocation, dataClone.ReportLocation, "This should be the same as the clone was just created");
            dataClone.ReportLocation = validUrl2;

            Assert.AreEqual("repo:/test/DotNetViewer4Pages.rpt", dataOriginal["report"], "Should be the one as defined in the beginning");            
            Assert.AreEqual("repo:/statistik/milestonetickets.rpt", dataClone["report"], "Should be the url that was set after the clone was created");

            // With Params
            dataOriginal["KeyOne"] = "ValueOne";                     
            Assert.AreEqual("ValueOne", dataOriginal["KeyOne"], "Test with param original");
               
            dataClone["KeyTwo"] = "ValueTwo";
            Assert.IsNull(dataClone["KeyOne"], "To make sure the param from the original was not used");
            Assert.AreEqual("ValueTwo", dataClone["KeyTwo"], "Test with param clone");            
        }

        /// <summary>
        /// 
        /// </summary>
        [Test]
        public void TestCloningPageCount()
        {
            IRenderData dataOriginal = new URLRenderData(validUrl);
            ReportViewShowError viewOriginal = new ReportViewShowError(dataOriginal, false);
            viewOriginal.CurrentPage = 4;
            viewOriginal.AssertException(3);

            IRenderData dataClone = dataOriginal.Clone() as IRenderData; // Subreport ondemand link clicked
            ReportViewShowError viewClone = new ReportViewShowError(dataClone);

            // Without params
            // this report only has 4 pages!
            Assert.AreEqual(4, viewOriginal.CurrentPage,  "Page should be 4");
            Assert.AreEqual(1, viewClone.CurrentPage, "Page should be 1");
            
            // change clone to 2 
            viewClone.Invoke((MethodInvoker)delegate
            {
                viewClone.CurrentPage = 2;                
            });
            Assert.AreEqual(4, viewOriginal.CurrentPage, "Page should be 4");
            Assert.AreEqual(2, viewClone.CurrentPage, "Page should be 2");
            
            // change orginal to 1
            viewClone.Invoke((MethodInvoker)delegate
            {
                viewOriginal.CurrentPage = 3;
            });
            Assert.AreEqual(3, viewOriginal.CurrentPage, "Page should be 1");
            Assert.AreEqual(2, viewClone.CurrentPage, "Page should be 2");
        }

        /// <summary>
        /// 
        /// </summary>
        [Test]
        public void TestReportWithParameters()
        {
            string url = validUrl + "&init=pdf&page=5&testParam1=testValue1";
            URLRenderData data = new URLRenderData(url);

            Assert.AreEqual("testValue1", data["testparam1"], "testParam1 in " + url);
            Assert.AreEqual(null , data["init"], "Init=java should be ignored " + url);
        }

        /// <summary>
        /// 
        /// </summary>
        [Test]
        public void TestReportWithPrompts()
        {            
            URLRenderData data = new URLRenderData(validSubreportOnDemand);            
        }


        /// <summary>
        /// 
        /// </summary>
        [Test]
        public void TestGetParametersFromUrl()
        {
            string url = "http://localhost:9000/?report=file:C:/Users/michaels/Documents/Group1.rpt&init=html&page=3&test1=test1value";
            URLRenderData data = new URLRenderData(url);

            Assert.AreEqual(null , data["init"], "html ignored");
            Assert.AreEqual(null , data["page"], "page ignored");
            Assert.AreEqual("test1value", data["test1"],  "test1"); 
        }              

        /// <summary>
        /// 
        /// </summary>
        [Test]
        public void TestGetParametersFromUrlDoubleParams()
        {
            // with double values always the last one in the url will be used.
            string url = "http://localhost:9000/?report=file:C:/Users/michaels/Documents/Group1.rpt&init=html&page=1.html&test1=test1value&test1=test1newvalue";
            URLRenderData data = new URLRenderData(url);

            Assert.AreEqual(null, data["init"], "html");
            Assert.AreEqual(null, data["page"], "page ");
            Assert.AreEqual("test1newvalue", data["test1"], "test1");                 
        }

        /// <summary>
        /// 
        /// </summary>
        [Test]
        public void TestStop()
        {
            string url = validServerUrlWithParameters;
            URLRenderData data = new URLRenderData(url);
            Byte[] pageData = data.GetPageData(1, false);
            Console.WriteLine("TestStop() pageData: " +pageData.Length);
            data.Stop();

            Byte[] pageDataAfterStop = data.GetPageData(2, false);
            Console.WriteLine("TestStop() pageDataAfterStop: " + pageDataAfterStop.Length);

            Assert.AreNotEqual(pageData.Length, pageDataAfterStop.Length, "The page size has to be different");
            
        }           

        /// <summary>
        /// This test is important for the promptonrefresh parameter
        /// </summary>
        [Test]      
        public void ChangeReportLocationTest()
        {
            // Without ReportView
            IRenderData data = new URLRenderData();
            data.ReportLocation = validServerUrlWithParameters;            
            data.ReportLocation = validServerUrlWithParameters + "&hasreportbutton=false&hasrefreshbutton=false";

            // With ReportView
            data = new URLRenderData();
            data.ReportLocation = validServerUrlWithParameters;
            ReportView view = new ReportView();
            view.ReportData = data;
            //data = new URLRenderData();
            // should not cause an exception
            data.ReportLocation = validServerUrlWithParameters + "&hasreportbutton=false&hasrefreshbutton=false";
        }
        
        /*
        public void TestExportChunk()
        {
                 IReportData data = new URLRenderData("http://server:9000/?report=C:/report.rpt");                 
                 FileStream fStream = new FileStream("C:/test.pdf", FileMode.OpenOrCreate);
                 data["export_fmt", "pdf" );
                 data[ "file", "c:/test.pdf" );
                 int count = data.GetExportChunkCount( props );
                 for(int i=1; count == 0 || i  < count; i++)
                 { 
                     // count can be 0 if there is gzip compression
                     byte[] pageData = data.GetExportChunkCount();
                     if( pageData != null ) 
                     {
                         fStream.Write( pageData );
                     } else 
                     {
                         break;
                     }
                 }
                 fStream.close();
        }
        */       
    }
}


